/*
 * File Name: main.c
 *
 * Description: The TRF797x is an integrated analog front end and
 * data framing system for a 13.56 MHz RFID reader system.
 * Built-in programming options make it suitable for a wide range
 * of applications both in proximity and vicinity RFID systems.
 * The reader is configured by selecting the desired protocol in
 * the control registers. Direct access to all control registers
 * allows fine tuning of various reader parameters as needed.
 *
 * The anti-collision procedures (as described in the ISO
 * standard 14443A) are implemented in the MCU firmware to help
 * the reader detect and communicate with one PICC among several
 * PICCs.
 *
 * Copyright (C) 2014 Texas Instruments Incorporated - http://www.ti.com/
 *
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/

#include "nfc.h"
#include "trf797x.h"
#include "mcu.h"
#include "uart.h"

//===============================================================

const static uint8_t CustomKey1[16] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
const static uint8_t CustomKey2[16] = {0xFA, 0xB4, 0xFA, 0xB4, 0xFA, 0xB4, 0xFA, 0xB4, 0xFA, 0xB4, 0xFA, 0xB4, 0xFA, 0xB4, 0xFA, 0xB4};
const static uint8_t CustomKey3[16] = {0x79, 0x70, 0x25, 0x53, 0x79, 0x70, 0x25, 0x53, 0x79, 0x70, 0x25, 0x53, 0x79, 0x70, 0x25, 0x53};;

//===============================================================
//
//===============================================================

void main(void)
{
	// Stop the Watchdog timer
	WDTCTL = WDTPW + WDTHOLD;
	
	// Select DCO to be 8 MHz
	McuOscSel(0x00);
	McuDelayMillisecond(10);

	// Set the SPI SS high
	SLAVE_SELECT_PORT_SET;
	SLAVE_SELECT_HIGH;

	// Set TRF Enable Pin high
	TRF_ENABLE_SET;
	TRF_ENABLE;

	// Wait until TRF system clock started
	McuDelayMillisecond(5);

	// Initialize SPI settings for communication with TRF
	Trf797xCommunicationSetup();

#ifdef ENABLE_UART
	// Set up UART
	UartSetup();
#endif

	// Set up TRF initial settings
	Trf797xInitialSettings();
	
	// Enable global interrupts
	__bis_SR_register(GIE);

    // Initial settings now complete
#ifdef ENABLE_UART
	UartTrfDFReaderMsg();
#endif

	while(1)
	{	
		// Set AES Key to a custom key stored in Flash Memory
		Nfc_setAesKey(CustomKey1,sizeof(CustomKey1));
		// Runs Authentication sequence
		Nfc_runAesAuth();

		// Repeat for the remaining keys
		Nfc_setAesKey(CustomKey2,sizeof(CustomKey2));
		Nfc_runAesAuth();

		Nfc_setAesKey(CustomKey3,sizeof(CustomKey3));
		Nfc_runAesAuth();
	}
}
